
Description: PS/2 mouse driver with packet processing
Language: c
Lines: 445

/*
 * SMACKTM Mouse Driver
 * PS/2 mouse driver with scroll wheel support
 */

#include "../kernel.h"
#include "mouse.h"

#define MOUSE_DATA_PORT     0x60
#define MOUSE_COMMAND_PORT  0x64
#define MOUSE_STATUS_PORT   0x64

// Mouse state
static struct mouse_state {
    int x, y;
    bool left_button;
    bool right_button;  
    bool middle_button;
    int scroll_delta;
    bool packet_ready;
    uint8_t packet[4];
    int packet_byte;
} mouse;

void mouse_init(void) {
    kprintf("Initializing PS/2 mouse driver...\n");
    
    // Reset mouse state
    mouse.x = 400;
    mouse.y = 300;
    mouse.left_button = false;
    mouse.right_button = false;
    mouse.middle_button = false;
    mouse.scroll_delta = 0;
    mouse.packet_ready = false;
    mouse.packet_byte = 0;
    
    // Enable mouse
    mouse_enable();
    
    // Register interrupt handler
    register_interrupt_handler(44, mouse_interrupt_handler);
    
    kprintf("Mouse driver initialized\n");
}

void mouse_interrupt_handler(void) {
    uint8_t data = inb(MOUSE_DATA_PORT);
    
    mouse.packet[mouse.packet_byte] = data;
    mouse.packet_byte++;
    
    if (mouse.packet_byte >= 3) {
        process_mouse_packet();
        mouse.packet_byte = 0;
    }
}

static void process_mouse_packet(void) {
    uint8_t flags = mouse.packet[0];
    int8_t dx = mouse.packet[1];
    int8_t dy = mouse.packet[2];
    
    // Update button states
    mouse.left_button = (flags & 0x01) != 0;
    mouse.right_button = (flags & 0x02) != 0;
    mouse.middle_button = (flags & 0x04) != 0;
    
    // Update position
    mouse.x += dx;
    mouse.y -= dy; // Invert Y axis
    
    // Clamp to screen bounds
    if (mouse.x < 0) mouse.x = 0;
    if (mouse.y < 0) mouse.y = 0;
    if (mouse.x >= 800) mouse.x = 799;
    if (mouse.y >= 600) mouse.y = 599;
    
    mouse.packet_ready = true;
}

void get_mouse_position(int* x, int* y) {
    *x = mouse.x;
    *y = mouse.y;
}

bool is_mouse_button_pressed(int button) {
    switch (button) {
        case 0: return mouse.left_button;
        case 1: return mouse.right_button;
        case 2: return mouse.middle_button;
        default: return false;
    }
}

================================================================================

