
Description: Programmable Interval Timer driver for system timing
Language: c
Lines: 378

/*
 * SMACKTM Timer Driver
 * PIT (Programmable Interval Timer) driver for system timing
 */

#include "../kernel.h"
#include "timer.h"

#define PIT_COMMAND_PORT  0x43
#define PIT_DATA_PORT_0   0x40
#define PIT_FREQUENCY     1193180

static volatile uint32_t timer_ticks = 0;
static uint32_t timer_frequency = 100; // 100 Hz default

void timer_init(uint32_t frequency) {
    kprintf("Initializing system timer (%d Hz)...\n", frequency);
    
    timer_frequency = frequency;
    timer_ticks = 0;
    
    // Calculate divisor
    uint32_t divisor = PIT_FREQUENCY / frequency;
    
    // Send command byte
    outb(PIT_COMMAND_PORT, 0x36);
    
    // Send frequency divisor
    outb(PIT_DATA_PORT_0, divisor & 0xFF);
    outb(PIT_DATA_PORT_0, (divisor >> 8) & 0xFF);
    
    // Register interrupt handler
    register_interrupt_handler(32, timer_interrupt_handler);
    
    kprintf("Timer initialized\n");
}

void timer_interrupt_handler(void) {
    timer_ticks++;
    
    // Call scheduler every 10ms (if frequency is 100Hz)
    if (timer_ticks % (timer_frequency / 100) == 0) {
        schedule();
    }
}

uint32_t get_timer_ticks(void) {
    return timer_ticks;
}

uint32_t get_uptime_seconds(void) {
    return timer_ticks / timer_frequency;
}

void timer_sleep(uint32_t milliseconds) {
    uint32_t target_ticks = timer_ticks + (milliseconds * timer_frequency / 1000);
    
    while (timer_ticks < target_ticks) {
        asm volatile("hlt");
    }
}

================================================================================

