
Description: VGA text mode display driver for console output
Language: c
Lines: 567

/*
 * SMACKTM VGA Display Driver
 * Text mode VGA driver for console output
 */

#include "../kernel.h"
#include "vga.h"

// VGA hardware constants
#define VGA_MEMORY       0xB8000
#define VGA_WIDTH        80
#define VGA_HEIGHT       25
#define VGA_SIZE         (VGA_WIDTH * VGA_HEIGHT)

// Current cursor position
static size_t vga_row;
static size_t vga_column;
static uint8_t vga_color;
static uint16_t* vga_buffer;

/*
 * Initialize VGA driver
 */
void vga_init(void) {
    vga_row = 0;
    vga_column = 0;
    vga_color = vga_entry_color(VGA_COLOR_LIGHT_GREY, VGA_COLOR_BLACK);
    vga_buffer = (uint16_t*)VGA_MEMORY;
    
    // Clear screen
    vga_clear();
    
    // Enable cursor
    vga_enable_cursor(14, 15);
    vga_update_cursor(0, 0);
}

/*
 * Create VGA color attribute
 */
uint8_t vga_entry_color(enum vga_color fg, enum vga_color bg) {
    return fg | bg << 4;
}

/*
 * Create VGA character entry
 */
uint16_t vga_entry(unsigned char uc, uint8_t color) {
    return (uint16_t)uc | (uint16_t)color << 8;
}

/*
 * Clear entire screen
 */
void vga_clear(void) {
    for (size_t i = 0; i < VGA_SIZE; i++) {
        vga_buffer[i] = vga_entry(' ', vga_color);
    }
    vga_row = 0;
    vga_column = 0;
    vga_update_cursor(vga_column, vga_row);
}

/*
 * Put character at specific position
 */
void vga_putentryat(char c, uint8_t color, size_t x, size_t y) {
    const size_t index = y * VGA_WIDTH + x;
    vga_buffer[index] = vga_entry(c, color);
}

/*
 * Put single character
 */
void vga_putchar(char c) {
    switch (c) {
        case '\n':
            vga_column = 0;
            if (++vga_row == VGA_HEIGHT) {
                vga_scroll();
            }
            break;
            
        case '\r':
            vga_column = 0;
            break;
            
        case '\t':
            vga_column = (vga_column + 8) & ~7;
            if (vga_column >= VGA_WIDTH) {
                vga_column = 0;
                if (++vga_row == VGA_HEIGHT) {
                    vga_scroll();
                }
            }
            break;
            
        default:
            vga_putentryat(c, vga_color, vga_column, vga_row);
            if (++vga_column == VGA_WIDTH) {
                vga_column = 0;
                if (++vga_row == VGA_HEIGHT) {
                    vga_scroll();
                }
            }
            break;
    }
    
    vga_update_cursor(vga_column, vga_row);
}

/*
 * Print string
 */
void vga_write(const char* data, size_t size) {
    for (size_t i = 0; i < size; i++) {
        vga_putchar(data[i]);
    }
}

/*
 * Print null-terminated string
 */
void vga_writestring(const char* data) {
    while (*data) {
        vga_putchar(*data++);
    }
}

================================================================================

