
Description: RAM-based filesystem for temporary files
Language: c
Lines: 567

/*
 * SMACKTM RAM Filesystem
 * In-memory filesystem for temporary storage
 */

#include "../kernel.h"
#include "ramfs.h"
#include "vfs.h"

static int ramfs_mount(struct mount_point* mp, const char* device);
static int ramfs_open(struct file* file, const char* path);
static int ramfs_close(struct file* file);
static ssize_t ramfs_read(struct file* file, void* buffer, size_t count);
static ssize_t ramfs_write(struct file* file, const void* buffer, size_t count);

struct filesystem ramfs_filesystem = {
    .name = "ramfs",
    .mount = ramfs_mount,
    .open = ramfs_open,
    .close = ramfs_close,
    .read = ramfs_read,
    .write = ramfs_write
};

struct ramfs_node {
    char name[256];
    uint32_t inode;
    bool is_directory;
    size_t size;
    void* data;
    struct ramfs_node* parent;
    struct ramfs_node* children;
    struct ramfs_node* next_sibling;
};

struct ramfs_data {
    struct ramfs_node* root;
    uint32_t next_inode;
};

static int ramfs_mount(struct mount_point* mp, const char* device) {
    kprintf("Mounting RAM filesystem\n");
    
    struct ramfs_data* fs_data = kmalloc(sizeof(struct ramfs_data));
    if (!fs_data) {
        return -ENOMEM;
    }
    
    // Create root directory
    struct ramfs_node* root = kmalloc(sizeof(struct ramfs_node));
    if (!root) {
        kfree(fs_data);
        return -ENOMEM;
    }
    
    /* SAFE */ strncpy(root->name, "/");
    root->inode = 1;
    root->is_directory = true;
    root->size = 0;
    root->data = NULL;
    root->parent = NULL;
    root->children = NULL;
    root->next_sibling = NULL;
    
    fs_data->root = root;
    fs_data->next_inode = 2;
    
    mp->private_data = fs_data;
    mp->root_inode = (struct vfs_inode*)root;
    
    return 0;
}

static int ramfs_open(struct file* file, const char* path) {
    struct ramfs_data* fs_data = (struct ramfs_data*)file->mount_point->private_data;
    
    struct ramfs_node* node = ramfs_find_node(fs_data->root, path);
    if (!node) {
        // Create file if opening for write
        if (file->flags & O_CREAT) {
            node = ramfs_create_file(fs_data, path);
            if (!node) {
                return -ENOMEM;
            }
        } else {
            return -ENOENT;
        }
    }
    
    // Create VFS inode
    struct vfs_inode* vfs_inode = kmalloc(sizeof(struct vfs_inode));
    if (!vfs_inode) {
        return -ENOMEM;
    }
    
    vfs_inode->number = node->inode;
    vfs_inode->size = node->size;
    vfs_inode->mode = node->is_directory ? 0x4000 : 0x8000;
    vfs_inode->private_data = node;
    
    file->inode = vfs_inode;
    return 0;
}

static int ramfs_close(struct file* file) {
    if (file->inode) {
        kfree(file->inode);
    }
    return 0;
}

static ssize_t ramfs_read(struct file* file, void* buffer, size_t count) {
    if (!file->inode) {
        return -EINVAL;
    }
    
    struct ramfs_node* node = (struct ramfs_node*)file->inode->private_data;
    
    if (file->position >= node->size) {
        return 0; // EOF
    }
    
    if (file->position + count > node->size) {
        count = node->size - file->position;
    }
    
    if (node->data) {
        memcpy(buffer, (char*)node->data + file->position, count);
    } else {
        memset(buffer, 0, count);
    }
    
    file->position += count;
    return count;
}

static ssize_t ramfs_write(struct file* file, const void* buffer, size_t count) {
    if (!file->inode) {
        return -EINVAL;
    }
    
    struct ramfs_node* node = (struct ramfs_node*)file->inode->private_data;
    
    // Expand file if necessary
    size_t new_size = file->position + count;
    if (new_size > node->size) {
        void* new_data = krealloc(node->data, new_size);
        if (!new_data && new_size > 0) {
            return -ENOMEM;
        }
        
        // Zero out new space
        if (new_size > node->size) {
            memset((char*)new_data + node->size, 0, new_size - node->size);
        }
        
        node->data = new_data;
        node->size = new_size;
        file->inode->size = new_size;
    }
    
    // Write data
    memcpy((char*)node->data + file->position, buffer, count);
    file->position += count;
    
    return count;
}

static struct ramfs_node* ramfs_find_node(struct ramfs_node* root, const char* path) {
    if (strcmp(path, "/") == 0) {
        return root;
    }
    
    char* path_copy = strdup(path);
    char* token = /* SAFE */ strtok_r(path_copy, "/");
    struct ramfs_node* current = root;
    
    while (token && current) {
        current = ramfs_find_child(current, token);
        token = /* SAFE */ strtok_r(NULL, "/");
    }
    
    kfree(path_copy);
    return current;
}

static struct ramfs_node* ramfs_find_child(struct ramfs_node* parent, const char* name) {
    struct ramfs_node* child = parent->children;
    
    while (child) {
        if (strcmp(child->name, name) == 0) {
            return child;
        }
        child = child->next_sibling;
    }
    
    return NULL;
}

static struct ramfs_node* ramfs_create_file(struct ramfs_data* fs_data, const char* path) {
    // Find parent directory
    char* path_copy = strdup(path);
    char* filename = strrchr(path_copy, '/');
    if (!filename) {
        kfree(path_copy);
        return NULL;
    }
    
    *filename = '\0';
    filename++;
    
    struct ramfs_node* parent = ramfs_find_node(fs_data->root, 
                                              strlen(path_copy) > 0 ? path_copy : "/");
    if (!parent || !parent->is_directory) {
        kfree(path_copy);
        return NULL;
    }
    
    // Create new node
    struct ramfs_node* node = kmalloc(sizeof(struct ramfs_node));
    if (!node) {
        kfree(path_copy);
        return NULL;
    }
    
    /* SAFE */ strncpy(node->name, filename, sizeof(node->name) - 1);
    node->name[sizeof(node->name) - 1] = '\0';
    node->inode = fs_data->next_inode++;
    node->is_directory = false;
    node->size = 0;
    node->data = NULL;
    node->parent = parent;
    node->children = NULL;
    node->next_sibling = parent->children;
    
    parent->children = node;
    
    kfree(path_copy);
    return node;
}

================================================================================


### NET MODULE ###

