
Description: Virtual File System implementation with multiple filesystem support
Language: c
Lines: 892

/*
 * SMACKTM Virtual File System
 * Unified interface for multiple filesystem types
 */

#include "../kernel.h"
#include "vfs.h"

static struct filesystem* filesystems[MAX_FILESYSTEMS];
static struct mount_point* mount_points[MAX_MOUNT_POINTS];
static struct file* open_files[MAX_OPEN_FILES];
static int fs_count = 0;
static int mount_count = 0;

void vfs_init(void) {
    kprintf("Initializing Virtual File System...\n");
    
    // Clear arrays
    for (int i = 0; i < MAX_FILESYSTEMS; i++) {
        filesystems[i] = NULL;
    }
    
    for (int i = 0; i < MAX_MOUNT_POINTS; i++) {
        mount_points[i] = NULL;
    }
    
    for (int i = 0; i < MAX_OPEN_FILES; i++) {
        open_files[i] = NULL;
    }
    
    fs_count = 0;
    mount_count = 0;
    
    // Register built-in filesystems
    register_filesystem(&ext2_filesystem);
    register_filesystem(&fat32_filesystem);
    register_filesystem(&ramfs_filesystem);
    
    // Create root mount point
    create_mount_point("/", "ramfs", NULL);
    
    kprintf("VFS initialized with %d filesystems\n", fs_count);
}

int register_filesystem(struct filesystem* fs) {
    if (fs_count >= MAX_FILESYSTEMS) {
        return -ENOSPC;
    }
    
    filesystems[fs_count] = fs;
    fs_count++;
    
    kprintf("Registered filesystem: %s\n", fs->name);
    return 0;
}

int create_mount_point(const char* path, const char* fstype, const char* device) {
    if (mount_count >= MAX_MOUNT_POINTS) {
        return -ENOSPC;
    }
    
    // Find filesystem type
    struct filesystem* fs = NULL;
    for (int i = 0; i < fs_count; i++) {
        if (strcmp(filesystems[i]->name, fstype) == 0) {
            fs = filesystems[i];
            break;
        }
    }
    
    if (!fs) {
        return -ENODEV;
    }
    
    // Create mount point
    struct mount_point* mp = kmalloc(sizeof(struct mount_point));
    if (!mp) {
        return -ENOMEM;
    }
    
    /* SAFE */ strncpy(mp->path, path, sizeof(mp->path) - 1);
    mp->path[sizeof(mp->path) - 1] = '\0';
    mp->filesystem = fs;
    mp->device = device ? strdup(device) : NULL;
    mp->root_inode = NULL;
    
    // Mount the filesystem
    if (fs->mount) {
        int result = fs->mount(mp, device);
        if (result != 0) {
            kfree(mp);
            return result;
        }
    }
    
    mount_points[mount_count] = mp;
    mount_count++;
    
    kprintf("Mounted %s on %s (type %s)\n", device ? device : "none", path, fstype);
    return 0;
}

struct file* vfs_open(const char* path, int flags) {
    // Find mount point
    struct mount_point* mp = find_mount_point(path);
    if (!mp) {
        return NULL;
    }
    
    // Find free file descriptor
    int fd = -1;
    for (int i = 0; i < MAX_OPEN_FILES; i++) {
        if (open_files[i] == NULL) {
            fd = i;
            break;
        }
    }
    
    if (fd == -1) {
        return NULL; // Too many open files
    }
    
    // Create file structure
    struct file* file = kmalloc(sizeof(struct file));
    if (!file) {
        return NULL;
    }
    
    file->fd = fd;
    file->mount_point = mp;
    file->flags = flags;
    file->position = 0;
    file->inode = NULL;
    
    // Open file through filesystem
    if (mp->filesystem->open) {
        int result = mp->filesystem->open(file, path + strlen(mp->path));
        if (result != 0) {
            kfree(file);
            return NULL;
        }
    }
    
    open_files[fd] = file;
    return file;
}

int vfs_close(struct file* file) {
    if (!file || file->fd < 0 || file->fd >= MAX_OPEN_FILES) {
        return -EBADF;
    }
    
    // Close through filesystem
    if (file->mount_point->filesystem->close) {
        file->mount_point->filesystem->close(file);
    }
    
    open_files[file->fd] = NULL;
    kfree(file);
    return 0;
}

ssize_t vfs_read(struct file* file, void* buffer, size_t count) {
    if (!file || !buffer) {
        return -EINVAL;
    }
    
    if (file->mount_point->filesystem->read) {
        return file->mount_point->filesystem->read(file, buffer, count);
    }
    
    return -ENOSYS;
}

ssize_t vfs_write(struct file* file, const void* buffer, size_t count) {
    if (!file || !buffer) {
        return -EINVAL;
    }
    
    if (file->mount_point->filesystem->write) {
        return file->mount_point->filesystem->write(file, buffer, count);
    }
    
    return -ENOSYS;
}

off_t vfs_seek(struct file* file, off_t offset, int whence) {
    if (!file) {
        return -EBADF;
    }
    
    switch (whence) {
        case SEEK_SET:
            file->position = offset;
            break;
        case SEEK_CUR:
            file->position += offset;
            break;
        case SEEK_END:
            if (file->inode) {
                file->position = file->inode->size + offset;
            } else {
                return -EINVAL;
            }
            break;
        default:
            return -EINVAL;
    }
    
    if (file->position < 0) {
        file->position = 0;
    }
    
    return file->position;
}

static struct mount_point* find_mount_point(const char* path) {
    struct mount_point* best_match = NULL;
    size_t best_length = 0;
    
    for (int i = 0; i < mount_count; i++) {
        struct mount_point* mp = mount_points[i];
        size_t len = strlen(mp->path);
        
        if (strncmp(path, mp->path, len) == 0 && len > best_length) {
            best_match = mp;
            best_length = len;
        }
    }
    
    return best_match;
}

================================================================================

