
Description: Memory management subsystem with paging and allocation
Language: c
Lines: 1247

/*
 * SMACKTM Memory Management
 * Physical and virtual memory management with paging
 */

#include "../kernel.h"
#include "memory.h"

// Memory bitmap for physical pages
static uint8_t* memory_bitmap;
static uint32_t total_pages;
static uint32_t used_pages;
static uint32_t free_pages;

// Kernel heap management
static struct heap_block* heap_head;
static uint32_t heap_size;

// Page directory and tables
static uint32_t* kernel_page_directory;
static uint32_t* kernel_page_tables[1024];

/*
 * Initialize memory management system
 */
int mm_init(void) {
    kprintf("Initializing memory management...\n");
    
    // Detect total memory
    uint32_t total_mem = detect_memory();
    total_pages = total_mem / PAGE_SIZE;
    
    kprintf("Total memory: %d KB (%d pages)\n", 
            total_mem / 1024, total_pages);
    
    // Initialize physical memory bitmap
    if (init_physical_memory() != 0) {
        return -1;
    }
    
    // Initialize paging
    if (init_paging() != 0) {
        return -1;
    }
    
    // Initialize kernel heap
    if (init_heap() != 0) {
        return -1;
    }
    
    kprintf("Memory management initialized successfully\n");
    return 0;
}

/*
 * Allocate a physical page
 */
uint32_t alloc_page(void) {
    if (free_pages == 0) {
        return 0;  // Out of memory
    }
    
    // Find first free page
    for (uint32_t i = 0; i < total_pages; i++) {
        if (is_page_free(i)) {
            mark_page_used(i);
            free_pages--;
            used_pages++;
            return i * PAGE_SIZE;
        }
    }
    
    return 0;  // No free pages found
}

/*
 * Free a physical page
 */
void free_page(uint32_t addr) {
    uint32_t page = addr / PAGE_SIZE;
    
    if (page >= total_pages || is_page_free(page)) {
        return;  // Invalid page or already free
    }
    
    mark_page_free(page);
    free_pages++;
    used_pages--;
}

/*
 * Kernel heap allocation
 */
void* kmalloc(size_t size) {
    if (size == 0) {
        return NULL;
    }
    
    // Align size to 8 bytes
    size = (size + 7) & ~7;
    
    // Find suitable block
    struct heap_block* block = find_free_block(size);
    if (!block) {
        return NULL;
    }
    
    block->used = true;
    return (void*)((uint32_t)block + sizeof(struct heap_block));
}

/*
 * Free kernel heap memory
 */
void kfree(void* ptr) {
    if (!ptr) {
        return;
    }
    
    struct heap_block* block = (struct heap_block*)((uint32_t)ptr - sizeof(struct heap_block));
    
    if (!block->used) {
        return;  // Double free
    }
    
    block->used = false;
}

================================================================================

