
Description: Network stack with TCP/IP implementation
Language: c
Lines: 1234

/*
 * SMACKTM Network Stack
 * Complete TCP/IP network implementation
 */

#include "../kernel.h"
#include "network.h"
#include "ethernet.h"
#include "ip.h"
#include "tcp.h"
#include "udp.h"

static struct network_interface interfaces[MAX_NETWORK_INTERFACES];
static int interface_count = 0;
static struct socket* sockets[MAX_SOCKETS];
static int next_socket_id = 1;

void network_init(void) {
    kprintf("Initializing network stack...\n");
    
    // Clear interfaces array
    for (int i = 0; i < MAX_NETWORK_INTERFACES; i++) {
        memset(&interfaces[i], 0, sizeof(struct network_interface));
    }
    
    // Clear sockets array
    for (int i = 0; i < MAX_SOCKETS; i++) {
        sockets[i] = NULL;
    }
    
    interface_count = 0;
    next_socket_id = 1;
    
    // Initialize protocol handlers
    ethernet_init();
    ip_init();
    tcp_init();
    udp_init();
    
    // Scan for network devices
    scan_network_devices();
    
    kprintf("Network stack initialized with %d interfaces\n", interface_count);
}

static void scan_network_devices(void) {
    // Scan PCI bus for network cards
    for (int i = 0; i < get_pci_device_count(); i++) {
        struct pci_device* dev = get_pci_device(i);
        
        // Check for network controller class
        if (dev->class_code == 0x02) {
            kprintf("Found network device: %04x:%04x\n", dev->vendor_id, dev->device_id);
            
            // Try to initialize known network cards
            if (dev->vendor_id == 0x8086) {
                // Intel network cards
                init_intel_network_card(dev);
            } else if (dev->vendor_id == 0x10EC) {
                // Realtek network cards
                init_realtek_network_card(dev);
            } else if (dev->vendor_id == 0x14E4) {
                // Broadcom network cards
                init_broadcom_network_card(dev);
            }
        }
    }
}

int register_network_interface(struct network_interface* iface) {
    if (interface_count >= MAX_NETWORK_INTERFACES) {
        return -ENOSPC;
    }
    
    memcpy(&interfaces[interface_count], iface, sizeof(struct network_interface));
    interfaces[interface_count].id = interface_count;
    interface_count++;
    
    kprintf("Registered network interface %s (MAC: %02x:%02x:%02x:%02x:%02x:%02x)\n",
            iface->name,
            iface->mac_address[0], iface->mac_address[1], iface->mac_address[2],
            iface->mac_address[3], iface->mac_address[4], iface->mac_address[5]);
    
    return interface_count - 1;
}

void network_receive_packet(int interface_id, void* packet, size_t length) {
    if (interface_id >= interface_count) {
        return;
    }
    
    struct network_interface* iface = &interfaces[interface_id];
    
    // Update interface statistics
    iface->rx_packets++;
    iface->rx_bytes += length;
    
    // Process packet through protocol stack
    ethernet_receive_packet(iface, packet, length);
}

int network_send_packet(int interface_id, void* packet, size_t length, uint8_t* dest_mac) {
    if (interface_id >= interface_count) {
        return -EINVAL;
    }
    
    struct network_interface* iface = &interfaces[interface_id];
    
    if (!iface->send_packet) {
        return -ENOSYS;
    }
    
    // Update interface statistics
    iface->tx_packets++;
    iface->tx_bytes += length;
    
    return iface->send_packet(iface, packet, length, dest_mac);
}

struct socket* socket_create(int domain, int type, int protocol) {
    // Find free socket slot
    int socket_id = -1;
    for (int i = 0; i < MAX_SOCKETS; i++) {
        if (sockets[i] == NULL) {
            socket_id = i;
            break;
        }
    }
    
    if (socket_id == -1) {
        return NULL; // No free sockets
    }
    
    // Allocate socket structure
    struct socket* sock = kmalloc(sizeof(struct socket));
    if (!sock) {
        return NULL;
    }
    
    memset(sock, 0, sizeof(struct socket));
    sock->id = next_socket_id++;
    sock->domain = domain;
    sock->type = type;
    sock->protocol = protocol;
    sock->state = SOCKET_CLOSED;
    
    // Initialize protocol-specific data
    switch (type) {
        case SOCK_STREAM:
            if (tcp_socket_init(sock) != 0) {
                kfree(sock);
                return NULL;
            }
            break;
        case SOCK_DGRAM:
            if (udp_socket_init(sock) != 0) {
                kfree(sock);
                return NULL;
            }
            break;
        default:
            kfree(sock);
            return NULL;
    }
    
    sockets[socket_id] = sock;
    return sock;
}

int socket_bind(struct socket* sock, struct sockaddr* addr, socklen_t addrlen) {
    if (!sock || !addr) {
        return -EINVAL;
    }
    
    switch (sock->type) {
        case SOCK_STREAM:
            return tcp_socket_bind(sock, addr, addrlen);
        case SOCK_DGRAM:
            return udp_socket_bind(sock, addr, addrlen);
        default:
            return -ENOTSUP;
    }
}

int socket_listen(struct socket* sock, int backlog) {
    if (!sock) {
        return -EINVAL;
    }
    
    if (sock->type != SOCK_STREAM) {
        return -EOPNOTSUPP;
    }
    
    return tcp_socket_listen(sock, backlog);
}

struct socket* socket_accept(struct socket* sock, struct sockaddr* addr, socklen_t* addrlen) {
    if (!sock) {
        return NULL;
    }
    
    if (sock->type != SOCK_STREAM) {
        return NULL;
    }
    
    return tcp_socket_accept(sock, addr, addrlen);
}

int socket_connect(struct socket* sock, struct sockaddr* addr, socklen_t addrlen) {
    if (!sock || !addr) {
        return -EINVAL;
    }
    
    switch (sock->type) {
        case SOCK_STREAM:
            return tcp_socket_connect(sock, addr, addrlen);
        case SOCK_DGRAM:
            return udp_socket_connect(sock, addr, addrlen);
        default:
            return -ENOTSUP;
    }
}

ssize_t socket_send(struct socket* sock, const void* buffer, size_t length, int flags) {
    if (!sock || !buffer) {
        return -EINVAL;
    }
    
    switch (sock->type) {
        case SOCK_STREAM:
            return tcp_socket_send(sock, buffer, length, flags);
        case SOCK_DGRAM:
            return udp_socket_send(sock, buffer, length, flags);
        default:
            return -ENOTSUP;
    }
}

ssize_t socket_receive(struct socket* sock, void* buffer, size_t length, int flags) {
    if (!sock || !buffer) {
        return -EINVAL;
    }
    
    switch (sock->type) {
        case SOCK_STREAM:
            return tcp_socket_receive(sock, buffer, length, flags);
        case SOCK_DGRAM:
            return udp_socket_receive(sock, buffer, length, flags);
        default:
            return -ENOTSUP;
    }
}

int socket_close(struct socket* sock) {
    if (!sock) {
        return -EINVAL;
    }
    
    // Find socket in array and remove it
    for (int i = 0; i < MAX_SOCKETS; i++) {
        if (sockets[i] == sock) {
            sockets[i] = NULL;
            break;
        }
    }
    
    // Protocol-specific cleanup
    switch (sock->type) {
        case SOCK_STREAM:
            tcp_socket_close(sock);
            break;
        case SOCK_DGRAM:
            udp_socket_close(sock);
            break;
    }
    
    kfree(sock);
    return 0;
}

void network_timer_tick(void) {
    // Call protocol timer functions
    tcp_timer_tick();
    ip_timer_tick();
}

struct network_interface* get_network_interface(int id) {
    if (id >= 0 && id < interface_count) {
        return &interfaces[id];
    }
    return NULL;
}

int get_network_interface_count(void) {
    return interface_count;
}

================================================================================

