
Description: Process management with creation, termination and IPC
Language: c
Lines: 756

/*
 * SMACKTM Process Management
 * Process creation, termination, and inter-process communication
 */

#include "../kernel.h"
#include "process.h"
#include "scheduler.h"

static struct process* processes[MAX_PROCESSES];
static uint32_t next_pid = 1;
static int process_count = 0;

void process_init(void) {
    kprintf("Initializing process management...\n");
    
    // Clear process table
    for (int i = 0; i < MAX_PROCESSES; i++) {
        processes[i] = NULL;
    }
    
    next_pid = 1;
    process_count = 0;
    
    kprintf("Process management initialized\n");
}

struct process* create_process(const char* name) {
    if (process_count >= MAX_PROCESSES) {
        return NULL;
    }
    
    // Find free slot
    int slot = -1;
    for (int i = 0; i < MAX_PROCESSES; i++) {
        if (processes[i] == NULL) {
            slot = i;
            break;
        }
    }
    
    if (slot == -1) {
        return NULL;
    }
    
    // Allocate process structure
    struct process* proc = kmalloc(sizeof(struct process));
    if (!proc) {
        return NULL;
    }
    
    memset(proc, 0, sizeof(struct process));
    
    // Initialize basic fields
    proc->pid = next_pid++;
    proc->state = PROC_READY;
    proc->priority = DEFAULT_PRIORITY;
    /* SAFE */ strncpy(proc->name, name, sizeof(proc->name) - 1);
    proc->name[sizeof(proc->name) - 1] = '\0';
    
    // Set parent process
    struct process* current = get_current_process();
    if (current) {
        proc->ppid = current->pid;
        proc->parent = current;
        
        // Add to parent's children list
        proc->next_sibling = current->children;
        current->children = proc;
    }
    
    // Allocate virtual memory space
    if (setup_process_memory(proc) != 0) {
        kfree(proc);
        return NULL;
    }
    
    // Initialize file descriptors
    for (int i = 0; i < MAX_OPEN_FILES; i++) {
        proc->files[i] = NULL;
    }
    
    // Add to process table
    processes[slot] = proc;
    process_count++;
    
    kprintf("Created process '%s' (PID %d)\n", name, proc->pid);
    return proc;
}

void destroy_process(struct process* proc) {
    if (!proc) {
        return;
    }
    
    kprintf("Destroying process '%s' (PID %d)\n", proc->name, proc->pid);
    
    // Remove from process table
    for (int i = 0; i < MAX_PROCESSES; i++) {
        if (processes[i] == proc) {
            processes[i] = NULL;
            process_count--;
            break;
        }
    }
    
    // Remove from parent's children list
    if (proc->parent) {
        struct process** child_ptr = &proc->parent->children;
        while (*child_ptr) {
            if (*child_ptr == proc) {
                *child_ptr = proc->next_sibling;
                break;
            }
            child_ptr = &(*child_ptr)->next_sibling;
        }
    }
    
    // Orphan children processes (assign to init)
    struct process* child = proc->children;
    while (child) {
        struct process* next_child = child->next_sibling;
        child->parent = get_process_by_pid(1); // init process
        child->ppid = 1;
        child = next_child;
    }
    
    // Clean up memory
    cleanup_process_memory(proc);
    
    // Free process structure
    kfree(proc);
}

struct process* get_process_by_pid(uint32_t pid) {
    for (int i = 0; i < MAX_PROCESSES; i++) {
        if (processes[i] && processes[i]->pid == pid) {
            return processes[i];
        }
    }
    return NULL;
}

int fork_process(void) {
    struct process* parent = get_current_process();
    if (!parent) {
        return -1;
    }
    
    // Create child process
    struct process* child = create_process(parent->name);
    if (!child) {
        return -1;
    }
    
    // Copy parent's memory space
    if (copy_process_memory(parent, child) != 0) {
        destroy_process(child);
        return -1;
    }
    
    // Copy file descriptors
    for (int i = 0; i < MAX_OPEN_FILES; i++) {
        child->files[i] = parent->files[i];
        if (child->files[i]) {
            // Increase reference count
            child->files[i]->ref_count++;
        }
    }
    
    // Copy registers (child will start from same point)
    child->esp = parent->esp;
    child->ebp = parent->ebp;
    child->eip = parent->eip;
    
    // Set return values
    // Parent gets child PID, child gets 0
    parent->eax = child->pid;
    child->eax = 0;
    
    // Add child to scheduler
    scheduler_add_process(child);
    
    return child->pid;
}

int exec_process(const char* path, char* const argv[], char* const envp[]) {
    struct process* proc = get_current_process();
    if (!proc) {
        return -1;
    }
    
    // Open executable file
    struct file* exe_file = vfs_open(path, O_RDONLY);
    if (!exe_file) {
        return -ENOENT;
    }
    
    // Load executable
    if (load_executable(proc, exe_file) != 0) {
        vfs_close(exe_file);
        return -ENOEXEC;
    }
    
    vfs_close(exe_file);
    
    // Set up command line arguments
    if (setup_process_args(proc, argv, envp) != 0) {
        return -ENOMEM;
    }
    
    // Update process name
    const char* filename = strrchr(path, '/');
    if (filename) {
        filename++;
    } else {
        filename = path;
    }
    
    /* SAFE */ strncpy(proc->name, filename, sizeof(proc->name) - 1);
    proc->name[sizeof(proc->name) - 1] = '\0';
    
    kprintf("Executed '%s' in process %d\n", path, proc->pid);
    
    // Jump to new program entry point
    // This doesn't return
    jump_to_user_mode(proc->eip, proc->esp);
    
    return 0; // Never reached
}

void exit_process(int exit_code) {
    struct process* proc = get_current_process();
    if (!proc) {
        return;
    }
    
    kprintf("Process '%s' (PID %d) exiting with code %d\n", 
            proc->name, proc->pid, exit_code);
    
    // Close all file descriptors
    for (int i = 0; i < MAX_OPEN_FILES; i++) {
        if (proc->files[i]) {
            vfs_close(proc->files[i]);
            proc->files[i] = NULL;
        }
    }
    
    // Notify scheduler
    scheduler_exit_process(proc, exit_code);
}

int wait_for_child(uint32_t pid, int* status) {
    struct process* parent = get_current_process();
    if (!parent) {
        return -1;
    }
    
    struct process* child = NULL;
    
    // Find specific child or any child
    if (pid == 0) {
        // Wait for any child
        child = parent->children;
        while (child && child->state != PROC_ZOMBIE) {
            child = child->next_sibling;
        }
    } else {
        // Wait for specific child
        child = get_process_by_pid(pid);
        if (!child || child->parent != parent) {
            return -ECHILD;
        }
    }
    
    // Block parent until child exits
    while (!child || child->state != PROC_ZOMBIE) {
        scheduler_block_process(parent, BLOCK_WAIT_CHILD);
        
        // Check again after being unblocked
        if (pid == 0) {
            child = parent->children;
            while (child && child->state != PROC_ZOMBIE) {
                child = child->next_sibling;
            }
        } else {
            child = get_process_by_pid(pid);
            if (!child) {
                return -ECHILD;
            }
        }
    }
    
    // Get exit status
    if (status) {
        *status = child->exit_code;
    }
    
    uint32_t child_pid = child->pid;
    
    // Clean up zombie child
    destroy_process(child);
    
    return child_pid;
}

void send_signal(uint32_t pid, int signal) {
    struct process* proc = get_process_by_pid(pid);
    if (!proc) {
        return;
    }
    
    // Set signal bit
    proc->signals |= (1 << signal);
    
    // Unblock process if it's blocked
    if (proc->state == PROC_BLOCKED) {
        scheduler_unblock_process(proc);
    }
}

static int setup_process_memory(struct process* proc) {
    // Allocate page directory
    proc->cr3 = alloc_page();
    if (proc->cr3 == 0) {
        return -ENOMEM;
    }
    
    // Initialize virtual memory areas
    proc->code_start = USER_BASE;
    proc->code_end = USER_BASE;
    proc->data_start = USER_BASE + 0x100000;
    proc->data_end = USER_BASE + 0x100000;
    proc->heap_start = USER_BASE + 0x200000;
    proc->heap_end = USER_BASE + 0x200000;
    proc->stack_start = USER_BASE + 0x800000 - KERNEL_STACK_SIZE;
    proc->stack_end = USER_BASE + 0x800000;
    
    // Allocate and map stack
    uint32_t stack_page = alloc_page();
    if (stack_page == 0) {
        free_page(proc->cr3);
        return -ENOMEM;
    }
    
    // Map stack page
    if (map_page(proc->cr3, proc->stack_start, stack_page, PAGE_PRESENT | PAGE_WRITABLE | PAGE_USER) != 0) {
        free_page(stack_page);
        free_page(proc->cr3);
        return -ENOMEM;
    }
    
    // Set initial stack pointer
    proc->esp = proc->stack_end - 4;
    proc->ebp = proc->esp;
    
    return 0;
}

void print_process_list(void) {
    kprintf("Process List:\n");
    kprintf("PID\tPPID\tState\t\tName\n");
    kprintf("---\t----\t-----\t\t----\n");
    
    for (int i = 0; i < MAX_PROCESSES; i++) {
        if (processes[i]) {
            struct process* proc = processes[i];
            const char* state_str;
            
            switch (proc->state) {
                case PROC_RUNNING: state_str = "RUNNING"; break;
                case PROC_READY: state_str = "READY"; break;
                case PROC_BLOCKED: state_str = "BLOCKED"; break;
                case PROC_ZOMBIE: state_str = "ZOMBIE"; break;
                default: state_str = "UNKNOWN"; break;
            }
            
            kprintf("%d\t%d\t%s\t\t%s\n", proc->pid, proc->ppid, state_str, proc->name);
        }
    }
    
    kprintf("Total processes: %d\n", process_count);
}

================================================================================


### COMPILATION INSTRUCTIONS ###


To compile the SMACKTM kernel:

1. Boot Loader (boot/boot.asm):
   nasm -f bin boot/boot.asm -o boot.bin

2. Multiboot Header (boot/multiboot.asm):
   nasm -f elf32 boot/multiboot.asm -o multiboot.o

3. Kernel C Files:
   gcc -m32 -c -ffreestanding -nostdlib kernel/main.c -o main.o
   gcc -m32 -c -ffreestanding -nostdlib mm/memory.c -o memory.o
   gcc -m32 -c -ffreestanding -nostdlib drivers/vga.c -o vga.o
   gcc -m32 -c -ffreestanding -nostdlib drivers/keyboard.c -o keyboard.o

4. Link kernel:
   ld -m elf_i386 -T linker.ld multiboot.o main.o memory.o vga.o keyboard.o -o kernel.bin

5. Create ISO image:
   grub-mkrescue -o smacktm.iso iso/

Required tools:
- NASM (Netwide Assembler)
- GCC cross-compiler for i386
- GNU LD linker
- GRUB utilities

Note: This is a basic 32-bit x86 kernel. For production use,
additional components like a file system, network stack, and
device drivers would need to be implemented.
