
Description: Process scheduler with round-robin and priority scheduling
Language: c
Lines: 823

/*
 * SMACKTM Process Scheduler
 * Multi-level priority scheduler with round-robin
 */

#include "../kernel.h"
#include "scheduler.h"
#include "process.h"

#define SCHEDULER_QUANTUM_MS    10
#define MAX_PRIORITY           20
#define MIN_PRIORITY           0
#define DEFAULT_PRIORITY       10

static struct process* ready_queues[MAX_PRIORITY + 1];
static struct process* current_process = NULL;
static struct process* idle_process = NULL;
static uint32_t quantum_ticks;
static uint32_t schedule_count = 0;

void scheduler_init(void) {
    kprintf("Initializing process scheduler...\n");
    
    // Clear ready queues
    for (int i = 0; i <= MAX_PRIORITY; i++) {
        ready_queues[i] = NULL;
    }
    
    current_process = NULL;
    quantum_ticks = 0;
    
    // Create idle process
    idle_process = create_idle_process();
    if (!idle_process) {
        kernel_panic("Failed to create idle process");
    }
    
    // Create init process
    struct process* init_proc = create_init_process();
    if (!init_proc) {
        kernel_panic("Failed to create init process");
    }
    
    // Add init to ready queue
    scheduler_add_process(init_proc);
    
    kprintf("Scheduler initialized\n");
}

void schedule(void) {
    // Save current process state if it exists
    if (current_process && current_process->state == PROC_RUNNING) {
        // Process used up its quantum, move to ready state
        current_process->state = PROC_READY;
        scheduler_add_process(current_process);
    }
    
    // Find next process to run
    struct process* next_process = scheduler_get_next_process();
    
    if (next_process == current_process) {
        // Same process, just reset quantum
        quantum_ticks = 0;
        return;
    }
    
    struct process* prev_process = current_process;
    current_process = next_process;
    
    if (current_process) {
        current_process->state = PROC_RUNNING;
        quantum_ticks = 0;
        schedule_count++;
        
        // Switch to new process
        if (prev_process) {
            context_switch(prev_process, current_process);
        } else {
            load_process_context(current_process);
        }
    }
}

void scheduler_add_process(struct process* proc) {
    if (!proc || proc->priority > MAX_PRIORITY) {
        return;
    }
    
    proc->state = PROC_READY;
    proc->next = NULL;
    
    // Add to appropriate priority queue
    if (ready_queues[proc->priority] == NULL) {
        ready_queues[proc->priority] = proc;
    } else {
        // Add to end of queue
        struct process* current = ready_queues[proc->priority];
        while (current->next) {
            current = current->next;
        }
        current->next = proc;
    }
}

void scheduler_remove_process(struct process* proc) {
    if (!proc || proc->priority > MAX_PRIORITY) {
        return;
    }
    
    struct process** queue = &ready_queues[proc->priority];
    
    if (*queue == proc) {
        *queue = proc->next;
    } else {
        struct process* current = *queue;
        while (current && current->next != proc) {
            current = current->next;
        }
        if (current) {
            current->next = proc->next;
        }
    }
    
    proc->next = NULL;
}

static struct process* scheduler_get_next_process(void) {
    // Check for processes in priority order (higher number = higher priority)
    for (int priority = MAX_PRIORITY; priority >= MIN_PRIORITY; priority--) {
        if (ready_queues[priority] != NULL) {
            struct process* proc = ready_queues[priority];
            ready_queues[priority] = proc->next;
            proc->next = NULL;
            return proc;
        }
    }
    
    // No ready processes, return idle
    return idle_process;
}

void scheduler_timer_interrupt(void) {
    quantum_ticks++;
    
    // Check if current process has used up its quantum
    if (current_process && quantum_ticks >= SCHEDULER_QUANTUM_MS) {
        schedule();
    }
    
    // Update process times
    if (current_process) {
        current_process->cpu_time++;
    }
}

void scheduler_block_process(struct process* proc, int reason) {
    if (!proc) {
        return;
    }
    
    proc->state = PROC_BLOCKED;
    proc->block_reason = reason;
    
    // Remove from ready queue if it's there
    scheduler_remove_process(proc);
    
    // If this is the current process, schedule another
    if (proc == current_process) {
        schedule();
    }
}

void scheduler_unblock_process(struct process* proc) {
    if (!proc || proc->state != PROC_BLOCKED) {
        return;
    }
    
    proc->state = PROC_READY;
    proc->block_reason = 0;
    scheduler_add_process(proc);
}

void scheduler_exit_process(struct process* proc, int exit_code) {
    if (!proc) {
        return;
    }
    
    proc->state = PROC_ZOMBIE;
    proc->exit_code = exit_code;
    
    // Remove from ready queue
    scheduler_remove_process(proc);
    
    // Close all open files
    for (int i = 0; i < MAX_OPEN_FILES; i++) {
        if (proc->files[i]) {
            vfs_close(proc->files[i]);
            proc->files[i] = NULL;
        }
    }
    
    // Notify parent process
    if (proc->parent) {
        scheduler_unblock_process(proc->parent);
    }
    
    // If this is the current process, schedule another
    if (proc == current_process) {
        current_process = NULL;
        schedule();
    }
}

int scheduler_set_priority(struct process* proc, int new_priority) {
    if (!proc || new_priority < MIN_PRIORITY || new_priority > MAX_PRIORITY) {
        return -EINVAL;
    }
    
    // Remove from current queue
    if (proc->state == PROC_READY) {
        scheduler_remove_process(proc);
    }
    
    // Update priority
    proc->priority = new_priority;
    
    // Re-add to appropriate queue
    if (proc->state == PROC_READY) {
        scheduler_add_process(proc);
    }
    
    return 0;
}

void yield(void) {
    if (current_process) {
        schedule();
    }
}

struct process* get_current_process(void) {
    return current_process;
}

static struct process* create_idle_process(void) {
    struct process* proc = kmalloc(sizeof(struct process));
    if (!proc) {
        return NULL;
    }
    
    memset(proc, 0, sizeof(struct process));
    proc->pid = 0;
    proc->ppid = 0;
    proc->state = PROC_READY;
    proc->priority = MIN_PRIORITY;
    /* SAFE */ strncpy(proc->name, "idle");
    
    // Set up idle process stack
    proc->stack_start = (uint32_t)kmalloc(KERNEL_STACK_SIZE);
    proc->stack_end = proc->stack_start + KERNEL_STACK_SIZE;
    proc->esp = proc->stack_end - 4;
    
    // Set up idle process entry point
    proc->eip = (uint32_t)idle_process_main;
    
    return proc;
}

static void idle_process_main(void) {
    while (1) {
        asm volatile("hlt");  // Halt until interrupt
    }
}

void print_scheduler_stats(void) {
    kprintf("Scheduler Statistics:\n");
    kprintf("  Total schedules: %d\n", schedule_count);
    kprintf("  Current process: %s (PID %d)\n", 
            current_process ? current_process->name : "none",
            current_process ? current_process->pid : 0);
    
    // Count processes in each queue
    for (int i = MAX_PRIORITY; i >= MIN_PRIORITY; i--) {
        int count = 0;
        struct process* proc = ready_queues[i];
        while (proc) {
            count++;
            proc = proc->next;
        }
        if (count > 0) {
            kprintf("  Priority %d: %d processes\n", i, count);
        }
    }
}

================================================================================

